require('dotenv').config();
const express = require('express');
const cors = require('cors');
const path = require('path');
const { initDatabase } = require('./db/init');

const app = express();
const PORT = process.env.PORT || 4000;

// Middleware
app.use(cors({
    origin: process.env.FRONTEND_URL || 'http://localhost:3000',
    credentials: true
}));
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// Static files for uploads
app.use('/uploads', express.static(path.join(__dirname, '../public/uploads')));

// Initialize database and start server
async function startServer() {
    try {
        await initDatabase();
        console.log('✅ Database initialized');

        // Import routes after database is ready
        const authRoutes = require('./routes/auth');
        const settingsRoutes = require('./routes/settings');
        const jobsRoutes = require('./routes/jobs');
        const applicationsRoutes = require('./routes/applications');

        // API Routes
        app.use('/api/auth', authRoutes);
        app.use('/api/settings', settingsRoutes);
        app.use('/api/jobs', jobsRoutes);
        app.use('/api/applications', applicationsRoutes);

        // Static file serving for resumes
        app.use('/uploads/resumes', express.static(path.join(__dirname, '../public/uploads/resumes')));

        // Health check
        app.get('/api/health', (req, res) => {
            res.json({ status: 'ok', timestamp: new Date().toISOString() });
        });

        // Error handling middleware
        app.use((err, req, res, next) => {
            console.error('Error:', err);
            res.status(err.status || 500).json({
                error: err.message || 'Internal server error'
            });
        });

        app.listen(PORT, () => {
            console.log(`🚀 Amralab CMS running on http://localhost:${PORT}`);
        });
    } catch (err) {
        console.error('Failed to start server:', err);
        process.exit(1);
    }
}

startServer();

module.exports = app;
