const express = require('express');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const { get, all, run } = require('../db/init');
const authMiddleware = require('../middleware/auth');

const router = express.Router();

// Ensure uploads directory exists
const uploadsDir = path.join(__dirname, '../../public/uploads/resumes');
if (!fs.existsSync(uploadsDir)) {
    fs.mkdirSync(uploadsDir, { recursive: true });
}

// Configure multer for file uploads
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, uploadsDir);
    },
    filename: (req, file, cb) => {
        // Generate unique filename: timestamp-randomstring-originalname
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        const ext = path.extname(file.originalname);
        const name = path.basename(file.originalname, ext);
        cb(null, `${name}-${uniqueSuffix}${ext}`);
    }
});

const fileFilter = (req, file, cb) => {
    // Accept only PDF, DOC, DOCX files
    const allowedTypes = [
        'application/pdf',
        'application/msword',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
    ];

    if (allowedTypes.includes(file.mimetype)) {
        cb(null, true);
    } else {
        cb(new Error('Invalid file type. Only PDF, DOC, and DOCX files are allowed.'), false);
    }
};

const upload = multer({
    storage: storage,
    fileFilter: fileFilter,
    limits: {
        fileSize: 5 * 1024 * 1024 // 5MB max file size
    }
});

// Get all applications (protected - admin only)
router.get('/', authMiddleware, (req, res) => {
    try {
        const { job_id, status } = req.query;

        let query = `
            SELECT 
                job_applications.*,
                jobs.title as job_title,
                jobs.department as job_department
            FROM job_applications
            LEFT JOIN jobs ON job_applications.job_id = jobs.id
            WHERE 1=1
        `;
        const params = [];

        if (job_id) {
            query += ' AND job_applications.job_id = ?';
            params.push(job_id);
        }

        if (status) {
            query += ' AND job_applications.status = ?';
            params.push(status);
        }

        query += ' ORDER BY job_applications.applied_at DESC';

        const applications = all(query, params);
        res.json(applications);
    } catch (err) {
        console.error('Error fetching applications:', err);
        res.status(500).json({ error: 'Failed to fetch applications' });
    }
});

// Get single application by ID (protected)
router.get('/:id', authMiddleware, (req, res) => {
    try {
        const application = get(`
            SELECT 
                job_applications.*,
                jobs.title as job_title,
                jobs.department as job_department
            FROM job_applications
            LEFT JOIN jobs ON job_applications.job_id = jobs.id
            WHERE job_applications.id = ?
        `, [req.params.id]);

        if (!application) {
            return res.status(404).json({ error: 'Application not found' });
        }

        res.json(application);
    } catch (err) {
        console.error('Error fetching application:', err);
        res.status(500).json({ error: 'Failed to fetch application' });
    }
});

// Get applications for a specific job (protected)
router.get('/job/:jobId', authMiddleware, (req, res) => {
    try {
        const applications = all(`
            SELECT * FROM job_applications 
            WHERE job_id = ? 
            ORDER BY applied_at DESC
        `, [req.params.jobId]);

        res.json(applications);
    } catch (err) {
        console.error('Error fetching job applications:', err);
        res.status(500).json({ error: 'Failed to fetch job applications' });
    }
});

// Submit application (public)
router.post('/', upload.single('resume'), (req, res) => {
    try {
        const { job_id, name, email, phone, cover_letter } = req.body;

        // Validate required fields
        if (!job_id || !name || !email || !req.file) {
            // Clean up uploaded file if validation fails
            if (req.file) {
                fs.unlinkSync(req.file.path);
            }
            return res.status(400).json({
                error: 'Missing required fields: job_id, name, email, and resume file'
            });
        }

        // Verify job exists
        const job = get('SELECT * FROM jobs WHERE id = ?', [job_id]);
        if (!job) {
            // Clean up uploaded file
            if (req.file) {
                fs.unlinkSync(req.file.path);
            }
            return res.status(404).json({ error: 'Job not found' });
        }

        // Check if job is still accepting applications
        if (!job.is_published) {
            if (req.file) {
                fs.unlinkSync(req.file.path);
            }
            return res.status(400).json({ error: 'This job is no longer accepting applications' });
        }

        // Store relative path for resume
        const resumeUrl = `/uploads/resumes/${req.file.filename}`;

        const result = run(`
            INSERT INTO job_applications (
                job_id, name, email, phone, cover_letter, resume_url, status
            )
            VALUES (?, ?, ?, ?, ?, ?, 'pending')
        `, [
            job_id,
            name,
            email,
            phone || '',
            cover_letter || '',
            resumeUrl
        ]);

        res.status(201).json({
            id: result.lastInsertRowid,
            message: 'Application submitted successfully'
        });
    } catch (err) {
        console.error('Error submitting application:', err);

        // Clean up uploaded file on error
        if (req.file) {
            try {
                fs.unlinkSync(req.file.path);
            } catch (unlinkErr) {
                console.error('Error deleting file:', unlinkErr);
            }
        }

        res.status(500).json({ error: 'Failed to submit application' });
    }
});

// Update application status (protected)
router.put('/:id/status', authMiddleware, (req, res) => {
    try {
        const { status } = req.body;

        // Validate status
        const validStatuses = ['pending', 'reviewing', 'shortlisted', 'rejected', 'hired'];
        if (!status || !validStatuses.includes(status)) {
            return res.status(400).json({
                error: `Invalid status. Must be one of: ${validStatuses.join(', ')}`
            });
        }

        const application = get('SELECT * FROM job_applications WHERE id = ?', [req.params.id]);
        if (!application) {
            return res.status(404).json({ error: 'Application not found' });
        }

        run(`
            UPDATE job_applications 
            SET status = ?
            WHERE id = ?
        `, [status, req.params.id]);

        res.json({ message: 'Application status updated successfully' });
    } catch (err) {
        console.error('Error updating application status:', err);
        res.status(500).json({ error: 'Failed to update application status' });
    }
});

// Update application notes (protected)
router.put('/:id/notes', authMiddleware, (req, res) => {
    try {
        const { notes } = req.body;

        const application = get('SELECT * FROM job_applications WHERE id = ?', [req.params.id]);
        if (!application) {
            return res.status(404).json({ error: 'Application not found' });
        }

        run(`
            UPDATE job_applications 
            SET notes = ?
            WHERE id = ?
        `, [notes || '', req.params.id]);

        res.json({ message: 'Application notes updated successfully' });
    } catch (err) {
        console.error('Error updating application notes:', err);
        res.status(500).json({ error: 'Failed to update application notes' });
    }
});

// Delete application (protected)
router.delete('/:id', authMiddleware, (req, res) => {
    try {
        const application = get('SELECT * FROM job_applications WHERE id = ?', [req.params.id]);

        if (!application) {
            return res.status(404).json({ error: 'Application not found' });
        }

        // Delete resume file
        if (application.resume_url) {
            const resumePath = path.join(__dirname, '../../public', application.resume_url);
            if (fs.existsSync(resumePath)) {
                try {
                    fs.unlinkSync(resumePath);
                } catch (err) {
                    console.error('Error deleting resume file:', err);
                }
            }
        }

        run('DELETE FROM job_applications WHERE id = ?', [req.params.id]);
        res.json({ message: 'Application deleted successfully' });
    } catch (err) {
        console.error('Error deleting application:', err);
        res.status(500).json({ error: 'Failed to delete application' });
    }
});

module.exports = router;
