const express = require('express');
const { get, all, run } = require('../db/init');
const authMiddleware = require('../middleware/auth');

const router = express.Router();

// Get all settings (public)
router.get('/', (req, res) => {
    try {
        const settings = all('SELECT * FROM settings');

        // Convert to key-value object
        const settingsObj = {};
        settings.forEach(s => {
            settingsObj[s.key] = s.value;
        });

        res.json(settingsObj);
    } catch (err) {
        res.status(500).json({ error: 'Failed to fetch settings' });
    }
});

// Get single setting
router.get('/:key', (req, res) => {
    try {
        const setting = get('SELECT * FROM settings WHERE key = ?', [req.params.key]);

        if (!setting) {
            return res.status(404).json({ error: 'Setting not found' });
        }

        res.json(setting);
    } catch (err) {
        res.status(500).json({ error: 'Failed to fetch setting' });
    }
});

// Update settings (protected) - accepts object of key-value pairs
router.put('/', authMiddleware, (req, res) => {
    try {
        const updates = req.body;

        for (const [key, value] of Object.entries(updates)) {
            run(`
        INSERT INTO settings (key, value, updated_at) 
        VALUES (?, ?, CURRENT_TIMESTAMP)
        ON CONFLICT(key) DO UPDATE SET value = excluded.value, updated_at = CURRENT_TIMESTAMP
      `, [key, value]);
        }

        res.json({ message: 'Settings updated successfully' });
    } catch (err) {
        console.error('Settings update error:', err);
        res.status(500).json({ error: 'Failed to update settings' });
    }
});

// Update single setting (protected)
router.put('/:key', authMiddleware, (req, res) => {
    try {
        const { value } = req.body;

        run(`
      INSERT INTO settings (key, value, updated_at) 
      VALUES (?, ?, CURRENT_TIMESTAMP)
      ON CONFLICT(key) DO UPDATE SET value = excluded.value, updated_at = CURRENT_TIMESTAMP
    `, [req.params.key, value]);

        res.json({ message: 'Setting updated successfully' });
    } catch (err) {
        res.status(500).json({ error: 'Failed to update setting' });
    }
});

module.exports = router;
